import tkinter as tk
from tkinter import messagebox, filedialog
from ftplib import FTP
import os

class FTPClientApp:
    def __init__(self, root):
        self.root = root
        self.root.title("FTP 文件夹传输客户端")

        # FTP 连接参数
        self.host = tk.StringVar()
        self.port = tk.StringVar(value="21")
        self.username = tk.StringVar(value="Administrator")
        self.password = tk.StringVar(value="1")

        # 文件夹路径
        self.local_folder_path = tk.StringVar()
        self.remote_folder_path = tk.StringVar(value="/boot")

        # 创建界面
        self.create_widgets()

    def create_widgets(self):
        # FTP 服务器地址
        tk.Label(self.root, text="FTP 服务器地址:").grid(row=0, column=0, padx=10, pady=5)
        tk.Entry(self.root, textvariable=self.host).grid(row=0, column=1, padx=10, pady=5)

        # 端口
        tk.Label(self.root, text="端口:").grid(row=1, column=0, padx=10, pady=5)
        tk.Entry(self.root, textvariable=self.port).grid(row=1, column=1, padx=10, pady=5)

        # 用户名
        tk.Label(self.root, text="用户名:").grid(row=2, column=0, padx=10, pady=5)
        tk.Entry(self.root, textvariable=self.username).grid(row=2, column=1, padx=10, pady=5)

        # 密码
        tk.Label(self.root, text="密码:").grid(row=3, column=0, padx=10, pady=5)
        tk.Entry(self.root, textvariable=self.password, show="*").grid(row=3, column=1, padx=10, pady=5)

        # 本地文件夹路径
        tk.Label(self.root, text="本地文件夹路径:").grid(row=4, column=0, padx=10, pady=5)
        tk.Entry(self.root, textvariable=self.local_folder_path).grid(row=4, column=1, padx=10, pady=5)
        tk.Button(self.root, text="选择文件夹", command=self.select_local_folder).grid(row=4, column=2, padx=10, pady=5)

        # 远程文件夹路径
        tk.Label(self.root, text="远程文件夹路径:").grid(row=5, column=0, padx=10, pady=5)
        tk.Entry(self.root, textvariable=self.remote_folder_path).grid(row=5, column=1, padx=10, pady=5)

        # 上传按钮
        tk.Button(self.root, text="上传文件夹", command=self.upload_folder).grid(row=6, column=0, padx=10, pady=10)

        # 下载按钮
        tk.Button(self.root, text="下载文件夹", command=self.download_folder).grid(row=6, column=1, padx=10, pady=10)

    def select_local_folder(self):
        """选择本地文件夹"""
        folder_path = filedialog.askdirectory()
        if folder_path:
            self.local_folder_path.set(folder_path)

    def connect_ftp(self):
        """连接到 FTP 服务器"""
        try:
            self.ftp = FTP()
            self.ftp.connect(self.host.get(), int(self.port.get()))
            self.ftp.login(self.username.get(), self.password.get())
            messagebox.showinfo("成功", "已成功连接到 FTP 服务器！")
            return True
        except Exception as e:
            messagebox.showerror("错误", f"连接失败: {e}")
            return False

    def upload_folder(self):
        """上传文件夹"""
        if not self.connect_ftp():
            return

        local_folder = self.local_folder_path.get()
        remote_folder = self.remote_folder_path.get()

        if not local_folder or not remote_folder:
            messagebox.showerror("错误", "请填写本地文件夹和远程文件夹路径！")
            return

        try:
            # 递归上传文件夹
            skipped_files = self._upload_folder_recursive(local_folder, remote_folder)
            if skipped_files:
                messagebox.showinfo("提示", f"文件夹上传完成，跳过了 {len(skipped_files)} 个空文件。")
            else:
                messagebox.showinfo("成功", "文件夹上传成功！")
        except Exception as e:
            messagebox.showerror("错误", f"文件夹上传失败: {e}")
        finally:
            self.ftp.quit()

    def _upload_folder_recursive(self, local_path, remote_path):
        """递归上传文件夹"""
        skipped_files = []  # 记录跳过的空文件

        # 创建远程文件夹
        try:
            self.ftp.mkd(remote_path)
        except Exception as e:
            # 如果文件夹已存在，忽略错误
            if "550" not in str(e):
                raise e

        # 遍历本地文件夹
        for item in os.listdir(local_path):
            local_item_path = os.path.join(local_path, item)
            remote_item_path = os.path.join(remote_path, item)
            print(item)
            
            if os.path.isfile(local_item_path):
                # 检查文件大小，跳过空文件
                if os.path.getsize(local_item_path) == 0:
                    with open(local_item_path, "rb") as file:
                        continue
                    skipped_files.append(local_item_path)
                    print(f"跳过空文件: {local_item_path}")
                    continue
                # 上传文件
                with open(local_item_path, "rb") as file:
                    self.ftp.storbinary(f"STOR {remote_item_path}", file)
            elif os.path.isdir(local_item_path):
                # 递归上传子文件夹
                self._upload_folder_recursive(local_item_path, remote_item_path)

        return skipped_files

    def download_folder(self):
        """下载文件夹"""
        if not self.connect_ftp():
            return

        local_folder = self.local_folder_path.get()
        remote_folder = self.remote_folder_path.get()

        if not local_folder or not remote_folder:
            messagebox.showerror("错误", "请填写本地文件夹和远程文件夹路径！")
            return

        try:
            # 递归下载文件夹
            skipped_files = self._download_folder_recursive(remote_folder, local_folder)
            if skipped_files:
                messagebox.showinfo("提示", f"文件夹下载完成，跳过了 {len(skipped_files)} 个空文件。")
            else:
                messagebox.showinfo("成功", "文件夹下载成功！")
        except Exception as e:
            messagebox.showerror("错误", f"文件夹下载失败: {e}")
        finally:
            self.ftp.quit()

    def _download_folder_recursive(self, remote_path, local_path):
        """递归下载文件夹"""
        skipped_files = []  # 记录跳过的空文件

        # 创建本地文件夹
        if not os.path.exists(local_path):
            os.makedirs(local_path)

        # 遍历远程文件夹
        for item in self.ftp.nlst(remote_path):
            remote_item_path = item
            local_item_path = os.path.join(local_path, os.path.basename(item))

            try:
                # 获取文件大小
                file_size = self.ftp.size(remote_item_path)
                if file_size == 0:
                    skipped_files.append(remote_item_path)
                    print(f"跳过空文件: {remote_item_path}")
                    continue
                # 下载文件
                with open(local_item_path, "wb") as file:
                    self.ftp.retrbinary(f"RETR {remote_item_path}", file.write)
            except Exception:
                # 如果是文件夹，递归下载子文件夹
                self._download_folder_recursive(remote_item_path, local_item_path)

        return skipped_files

    def delete_remote_folder_contents(self):
        """删除远程文件夹内容"""
        if not self.connect_ftp():
            return

        remote_folder = self.remote_folder_path.get()

        if not remote_folder:
            messagebox.showerror("错误", "请填写远程文件夹路径！")
            return

        try:
            # 递归删除远程文件夹内容
            self._delete_remote_folder_contents(remote_folder)
            messagebox.showinfo("成功", "远程文件夹内容已删除！")
        except Exception as e:
            messagebox.showerror("错误", f"删除远程文件夹内容失败: {e}")
        finally:
            self.ftp.quit()

# 创建主窗口
root = tk.Tk()
app = FTPClientApp(root)
root.mainloop()
